/*
  ==============================================================================

    SonicCrypt Chaos Engine
    Copyright (C) 2025 Sebastian Sünkler

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU Affero General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Affero General Public License for more details.

    You should have received a copy of the GNU Affero General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

  ==============================================================================
*/

#include "PluginProcessor.h"
#include "PluginEditor.h"

SonicCryptChaosAudioProcessor::SonicCryptChaosAudioProcessor()
#ifndef JucePlugin_PreferredChannelConfigurations
    : AudioProcessor(BusesProperties().withInput("Input", juce::AudioChannelSet::stereo(), true).withOutput("Output", juce::AudioChannelSet::stereo(), true)),
    apvts(*this, nullptr, "Parameters", createParameterLayout())
#endif
{
    formatManager.registerBasicFormats();
    synth.clearSounds(); synth.clearVoices();
    synth.addSound(new SonicCryptSound());
    for (int i = 0; i < maxVoices; i++) synth.addVoice(new SonicCryptVoice());
    currentMode = ChaosMode::Rhythm;
    rhythmPattern.fill(1.0f);
    currentOutputLevel = 0.0f;

    auto folder = getPresetFolder();
    if (!folder.exists()) folder.createDirectory();
}

SonicCryptChaosAudioProcessor::~SonicCryptChaosAudioProcessor() {}

juce::AudioProcessorValueTreeState::ParameterLayout SonicCryptChaosAudioProcessor::createParameterLayout()
{
    juce::AudioProcessorValueTreeState::ParameterLayout layout;
    layout.add(std::make_unique<juce::AudioParameterChoice>("mode", "Chaos Mode", juce::StringArray{ "OneShot", "Rhythm", "Drone" }, 1));
    layout.add(std::make_unique<juce::AudioParameterBool>("lfoOn", "Drift LFO", true));
    layout.add(std::make_unique<juce::AudioParameterChoice>("lfoRate", "LFO Rate", juce::StringArray{ "1/1", "1/2", "1/4", "1/8", "1/16", "1/32" }, 3));
    layout.add(std::make_unique<juce::AudioParameterFloat>("lfoDepth", "LFO Wobble", 0.0f, 1.0f, 0.0f));

    layout.add(std::make_unique<juce::AudioParameterFloat>("attack", "Attack", 0.0f, 5.0f, 0.1f));
    layout.add(std::make_unique<juce::AudioParameterFloat>("release", "Release", 0.01f, 10.0f, 0.5f));
    layout.add(std::make_unique<juce::AudioParameterFloat>("drive", "Distortion", 0.0f, 50.0f, 0.0f));
    layout.add(std::make_unique<juce::AudioParameterFloat>("cutoff", "Filter Freq", juce::NormalisableRange<float>(20.0f, 20000.0f, 1.0f, 0.2f), 20000.0f));
    layout.add(std::make_unique<juce::AudioParameterFloat>("resonance", "Filter Res", 0.0f, 1.0f, 0.0f));
    layout.add(std::make_unique<juce::AudioParameterFloat>("delayTime", "Delay Time", 0.01f, 1.0f, 0.5f));
    layout.add(std::make_unique<juce::AudioParameterFloat>("delayFeedback", "Delay Fdbk", 0.0f, 0.95f, 0.3f));
    layout.add(std::make_unique<juce::AudioParameterFloat>("reverbSize", "Reverb Size", 0.0f, 1.0f, 0.5f));
    layout.add(std::make_unique<juce::AudioParameterFloat>("reverbMix", "Reverb Mix", 0.0f, 1.0f, 0.3f));
    layout.add(std::make_unique<juce::AudioParameterFloat>("gain", "Master Gain", 0.0f, 2.0f, 1.0f));
    layout.add(std::make_unique<juce::AudioParameterChoice>("rhythmRate", "Rhythm Rate", juce::StringArray{ "1/4", "1/8", "1/16", "1/32" }, 2));

    return layout;
}

void SonicCryptChaosAudioProcessor::prepareToPlay(double sampleRate, int samplesPerBlock)
{
    synth.setCurrentPlaybackSampleRate(sampleRate);
    juce::dsp::ProcessSpec spec{ sampleRate, (juce::uint32)samplesPerBlock, (juce::uint32)getTotalNumOutputChannels() };
    distortionModule.prepare(spec); distortionModule.functionToUse = [](float x) { return std::tanh(x); };
    filterModule.prepare(spec); delayModule.prepare(spec); delayModule.setMaximumDelayInSamples(sampleRate * 2.0);
    reverbModule.prepare(spec); gainModule.prepare(spec); limiterModule.prepare(spec); limiterModule.setThreshold(-0.5f); limiterModule.setRelease(100.0f);
    lfoPhase = 0.0f; gatePhaseSamples = 0.0; lastNote = 60;
    delayTimeSmoothed.reset(sampleRate, 0.2);
}

void SonicCryptChaosAudioProcessor::releaseResources() {}

void SonicCryptChaosAudioProcessor::resetEffects() { delayModule.reset(); reverbModule.reset(); lfoPhase = 0.0f; gatePhaseSamples = 0.0; }

void SonicCryptChaosAudioProcessor::processBlock(juce::AudioBuffer<float>& buffer, juce::MidiBuffer& midiMessages)
{
    juce::ScopedNoDenormals noDenormals;
    buffer.clear();
    currentMode = static_cast<ChaosMode>(static_cast<int>(*apvts.getRawParameterValue("mode")));

    for (const auto metadata : midiMessages) {
        if (metadata.getMessage().isNoteOn()) {
            lfoPhase = 0.0f; gatePhaseSamples = 0.0; lastNote = metadata.getMessage().getNoteNumber();
        }
    }

    bool shouldLoop = (currentMode != ChaosMode::OneShot);
    juce::ADSR::Parameters adsrParams;
    adsrParams.attack = *apvts.getRawParameterValue("attack");
    adsrParams.decay = 0.0f; adsrParams.sustain = 1.0f;
    adsrParams.release = *apvts.getRawParameterValue("release");

    for (int i = 0; i < synth.getNumVoices(); ++i) {
        if (auto* voice = dynamic_cast<SonicCryptVoice*>(synth.getVoice(i))) {
            voice->setLooping(shouldLoop); voice->setADSR(adsrParams);
        }
    }

    { juce::ScopedLock sl(sampleLoadingLock); synth.renderNextBlock(buffer, midiMessages, 0, buffer.getNumSamples()); }

    juce::dsp::AudioBlock<float> block(buffer);
    juce::dsp::ProcessContextReplacing<float> context(block);

    double bpm = 120.0; double ppq = 0.0; bool isPlaying = false;
    if (auto* ph = getPlayHead()) {
        if (auto pos = ph->getPosition()) {
            isPlaying = pos->getIsPlaying();
            if (pos->getBpm().hasValue()) bpm = *pos->getBpm();
            if (pos->getPpqPosition().hasValue()) ppq = *pos->getPpqPosition();
        }
    }

    if (currentMode == ChaosMode::Rhythm) {
        int rateIdx = (int)*apvts.getRawParameterValue("rhythmRate");
        float currentGrid = 1.0f; if (rateIdx == 1) currentGrid = 2.0f; else if (rateIdx == 2) currentGrid = 4.0f; else if (rateIdx == 3) currentGrid = 8.0f;
        double samplesPerStep = (60.0 / bpm) * (1.0 / currentGrid) * getSampleRate();
        for (int c = 0; c < buffer.getNumChannels(); ++c) {
            auto* data = buffer.getWritePointer(c);
            double localGatePhase = gatePhaseSamples;
            for (int s = 0; s < buffer.getNumSamples(); ++s) {
                double effectivePos = isPlaying ? (ppq + (s / getSampleRate()) * (bpm / 60.0)) * currentGrid : (localGatePhase / samplesPerStep);
                int step = (int)std::floor(effectivePos) % 16;
                if (step < 0) step = 0; if (step > 15) step = 15;
                data[s] *= rhythmPattern[step];
                localGatePhase += 1.0;
            }
            if (c == 0 && !isPlaying) gatePhaseSamples += (double)buffer.getNumSamples();
        }
    }

    float lfoMod = 0.0f;
    if (*apvts.getRawParameterValue("lfoOn") > 0.5f) {
        int rateIndex = (int)*apvts.getRawParameterValue("lfoRate");
        double hz = bpm / 60.0; double rateFactor = 1.0;
        if (rateIndex == 1) { hz /= 4.0; rateFactor = 0.25; }
        else if (rateIndex == 2) { hz /= 2.0; rateFactor = 0.5; }
        else if (rateIndex == 4) { hz *= 2.0; rateFactor = 2.0; }
        else if (rateIndex == 5) { hz *= 4.0; rateFactor = 4.0; }
        else if (rateIndex == 6) { hz *= 8.0; rateFactor = 8.0; }

        if (isPlaying && currentMode != ChaosMode::OneShot) {
            double syncedPhase = ppq * rateFactor * 2.0 * juce::MathConstants<double>::pi;
            lfoMod = std::sin(syncedPhase) * (*apvts.getRawParameterValue("lfoDepth"));
        }
        else {
            double phaseInc = (hz * juce::MathConstants<double>::twoPi) / getSampleRate();
            lfoPhase += (float)(phaseInc * buffer.getNumSamples());
            if (lfoPhase > juce::MathConstants<float>::twoPi) lfoPhase -= juce::MathConstants<float>::twoPi;
            lfoMod = std::sin(lfoPhase) * (*apvts.getRawParameterValue("lfoDepth"));
        }
    }

    float drive = *apvts.getRawParameterValue("drive");
    if (drive > 0.1f) {
        float gain = std::pow(10.0f, drive / 20.0f);
        for (int c = 0; c < buffer.getNumChannels(); ++c) {
            auto* data = buffer.getWritePointer(c);
            for (int s = 0; s < buffer.getNumSamples(); ++s) data[s] = std::tanh(data[s] * gain);
        }
    }

    float cutoff = *apvts.getRawParameterValue("cutoff");
    float keyTrack = std::pow(2.0f, (lastNote - 60.0f) / 12.0f);
    cutoff *= keyTrack;
    if (std::abs(lfoMod) > 0.001f) cutoff *= std::pow(2.0f, lfoMod * 3.0f);
    cutoff = juce::jlimit(20.0f, 20000.0f, cutoff);
    filterModule.setCutoffFrequencyHz(cutoff);
    filterModule.setResonance(*apvts.getRawParameterValue("resonance"));
    filterModule.setMode(juce::dsp::LadderFilterMode::LPF12);
    filterModule.process(context);

    float targetDelay = *apvts.getRawParameterValue("delayTime") * getSampleRate();
    delayTimeSmoothed.setTargetValue(targetDelay);
    float delaySamps = delayTimeSmoothed.getNextValue();

    if (delaySamps > 0 && delaySamps < delayModule.getMaximumDelayInSamples()) {
        juce::AudioBuffer<float> dryBuffer; dryBuffer.makeCopyOf(buffer);
        delayModule.setDelay(delaySamps); delayModule.process(context);
        float fb = *apvts.getRawParameterValue("delayFeedback");
        for (int c = 0; c < buffer.getNumChannels(); ++c)
            buffer.addFrom(c, 0, dryBuffer, c, 0, buffer.getNumSamples(), 1.0f - (fb * 0.5f));
    }

    juce::dsp::Reverb::Parameters rp;
    rp.roomSize = *apvts.getRawParameterValue("reverbSize");
    rp.wetLevel = *apvts.getRawParameterValue("reverbMix");
    rp.dryLevel = 1.0f - rp.wetLevel;
    reverbModule.setParameters(rp); reverbModule.process(context);
    gainModule.setGainLinear(*apvts.getRawParameterValue("gain"));
    gainModule.process(context); limiterModule.process(context);

    // VISUALIZATION MEASUREMENT
    currentOutputLevel = buffer.getMagnitude(0, buffer.getNumSamples());
}

void SonicCryptChaosAudioProcessor::triggerRandomChaos()
{
    for (int i = 0; i < synth.getNumVoices(); ++i) if (auto* voice = synth.getVoice(i)) voice->stopNote(0.0f, false);
    resetEffects();
    juce::Random rng(juce::Time::currentTimeMillis());
    auto setParam = [&](juce::String id, float v) {
        if (auto* p = apvts.getParameter(id)) {
            auto r = p->getNormalisableRange();
            p->setValueNotifyingHost(r.convertTo0to1(juce::jlimit(r.start, r.end, v)));
        }
        };
    auto randRange = [&](float min, float max) { return min + rng.nextFloat() * (max - min); };

    rhythmPattern.fill(1.0f);
    if (currentMode == ChaosMode::Rhythm) {
        rhythmPattern.fill(0.0f); rhythmPattern[0] = 1.0f; rhythmPattern[8] = 1.0f;
        for (int i = 0; i < 16; i++) if (rng.nextFloat() > 0.6f) rhythmPattern[i] = 1.0f;
        setParam("attack", 0.001f); setParam("release", 0.15f);
        setParam("drive", randRange(15.0f, 40.0f));
        setParam("lfoDepth", randRange(0.4f, 0.8f));
        setParam("cutoff", randRange(500.0f, 5000.0f));
        setParam("resonance", randRange(0.2f, 0.6f));
        setParam("reverbMix", randRange(0.1f, 0.4f));
        setParam("delayTime", randRange(0.1f, 0.4f));
        setParam("delayFeedback", randRange(0.2f, 0.5f));
    }
    else if (currentMode == ChaosMode::Drone) {
        setParam("attack", randRange(1.5f, 4.0f)); setParam("release", randRange(4.0f, 9.0f));
        setParam("drive", randRange(2.0f, 15.0f)); setParam("lfoDepth", randRange(0.05f, 0.25f));
        setParam("cutoff", randRange(150.0f, 1500.0f)); setParam("resonance", randRange(0.5f, 0.9f));
        setParam("reverbSize", randRange(0.8f, 1.0f)); setParam("reverbMix", randRange(0.7f, 1.0f));
        setParam("delayTime", randRange(0.5f, 1.5f)); setParam("delayFeedback", randRange(0.6f, 0.9f));
    }
    else {
        setParam("attack", 0.0f); setParam("release", randRange(0.3f, 1.0f)); setParam("drive", randRange(10.0f, 50.0f));
        setParam("reverbMix", randRange(0.2f, 0.5f)); setParam("reverbSize", randRange(0.3f, 0.6f));
        setParam("delayTime", randRange(0.01f, 0.15f)); setParam("delayFeedback", randRange(0.1f, 0.3f));
        setParam("lfoDepth", 0.0f); setParam("resonance", randRange(0.0f, 0.4f)); setParam("cutoff", randRange(1000.0f, 15000.0f));
    }
    shuffleSamplesFromCache();
}

void SonicCryptChaosAudioProcessor::loadSamplesFromFolder(const juce::File& folder)
{
    if (!folder.isDirectory()) return;
    currentSampleFolder = folder;
    allSamplePaths.clear();
    juce::StringArray allowedFormats = { "*.wav", "*.aiff", "*.mp3" };
    for (auto entry : juce::RangedDirectoryIterator(folder, true, allowedFormats.joinIntoString(";")))
        allSamplePaths.add(entry.getFile().getFullPathName());
    shuffleSamplesFromCache();
}

void SonicCryptChaosAudioProcessor::loadSmartSliceIntoSound(SonicCryptSound* sound, int layerIndex, juce::File file)
{
    std::unique_ptr<juce::AudioFormatReader> reader(formatManager.createReaderFor(file));
    if (reader == nullptr) return;
    int maxSliceLen = 10 * (int)reader->sampleRate;
    int fileLen = (int)reader->lengthInSamples;
    int sliceLen = juce::jmin(fileLen, maxSliceLen);
    juce::Random rng(juce::Time::currentTimeMillis() + layerIndex);
    int startPos = 0;
    if (fileLen > sliceLen) startPos = rng.nextInt(fileLen - sliceLen);

    juce::AudioBuffer<float> finalBuffer((int)reader->numChannels, sliceLen);
    reader->read(&finalBuffer, 0, sliceLen, startPos, true, true);
    float magnitude = finalBuffer.getMagnitude(0, sliceLen);
    if (magnitude > 0.001f) { float gain = 0.95f / magnitude; finalBuffer.applyGain(gain); }

    float pan = 0.5f;
    if (layerIndex == 0) pan = 0.2f;
    else if (layerIndex == 1) pan = 0.8f;
    else if (layerIndex == 2) pan = 0.4f;
    else pan = 0.6f;

    sound->setLayer(layerIndex, finalBuffer, reader->sampleRate, pan);
}

void SonicCryptChaosAudioProcessor::shuffleSamplesFromCache()
{
    juce::ScopedLock sl(sampleLoadingLock);
    if (allSamplePaths.isEmpty()) return;
    auto* sound = dynamic_cast<SonicCryptSound*>(synth.getSound(0).get());
    if (!sound) return;
    sound->clear();
    juce::Random rng(juce::Time::currentTimeMillis());
    for (int i = 0; i < 4; ++i) {
        auto pathIndex = rng.nextInt(allSamplePaths.size());
        loadSmartSliceIntoSound(sound, i, juce::File(allSamplePaths[pathIndex]));
    }
}

juce::File SonicCryptChaosAudioProcessor::getPresetFolder() const {
    auto docDir = juce::File::getSpecialLocation(juce::File::userDocumentsDirectory);
    return docDir.getChildFile("SonicCrypt").getChildFile("ChaosEngine").getChildFile("Presets");
}
void SonicCryptChaosAudioProcessor::writeAudioLayerToStream(juce::OutputStream& out, SonicCryptSound* sound, int layerIndex) {
    auto& layer = sound->layers[layerIndex]; out.writeBool(layer.isValid);
    if (layer.isValid) {
        out.writeDouble(layer.sourceRate); out.writeInt(layer.buffer.getNumChannels()); out.writeInt(layer.buffer.getNumSamples());
        for (int c = 0; c < layer.buffer.getNumChannels(); ++c) out.write(layer.buffer.getReadPointer(c), layer.buffer.getNumSamples() * sizeof(float));
    }
}
void SonicCryptChaosAudioProcessor::readAudioLayerFromStream(juce::InputStream& in, SonicCryptSound* sound, int layerIndex) {
    if (in.readBool()) {
        double rate = in.readDouble(); int channels = in.readInt(); int samples = in.readInt();
        if (channels > 0 && samples > 0 && samples < 100 * 1024 * 1024) {
            juce::AudioBuffer<float> buf(channels, samples);
            for (int c = 0; c < channels; ++c) in.read(buf.getWritePointer(c), samples * sizeof(float));
            float pan = 0.5f; if (layerIndex == 0) pan = 0.2f; else if (layerIndex == 1) pan = 0.8f; else if (layerIndex == 2) pan = 0.4f; else pan = 0.6f;
            sound->setLayer(layerIndex, buf, rate, pan);
        }
        else { in.skipNextBytes(samples * channels * sizeof(float)); }
    }
}
void SonicCryptChaosAudioProcessor::savePreset(const juce::String& name) {
    auto folder = getPresetFolder(); if (!folder.exists()) folder.createDirectory();
    auto file = folder.getChildFile(name).withFileExtension("scp");
    juce::ScopedLock sl(sampleLoadingLock);
    juce::FileOutputStream out(file);
    if (out.openedOk()) {
        out.writeString("SCBIN4");
        juce::MemoryBlock paramBlock; {
            juce::MemoryOutputStream mos(paramBlock, false); auto state = apvts.copyState();
            state.setProperty("sampleFolder", currentSampleFolder.getFullPathName(), nullptr);
            juce::String patternString; for (float f : rhythmPattern) patternString += (f > 0.5f ? "1" : "0");
            state.setProperty("rhythmPattern", patternString, nullptr);
            state.writeToStream(mos);
        }
        out.writeInt64(paramBlock.getSize()); out.write(paramBlock.getData(), paramBlock.getSize());
        auto* sound = dynamic_cast<SonicCryptSound*>(synth.getSound(0).get());
        if (sound) { for (int i = 0; i < 4; ++i) writeAudioLayerToStream(out, sound, i); }
    }
}
void SonicCryptChaosAudioProcessor::loadPresetFile(const juce::File& file) {
    if (!file.existsAsFile()) return;
    juce::ScopedLock sl(sampleLoadingLock);
    juce::FileInputStream in(file);
    if (in.openedOk()) {
        if (in.readString() != "SCBIN4") return;
        juce::int64 paramSize = in.readInt64();
        if (paramSize > 0 && paramSize < 1024 * 1024) {
            juce::MemoryBlock paramBlock; in.readIntoMemoryBlock(paramBlock, (size_t)paramSize);
            juce::MemoryInputStream mis(paramBlock, false);
            auto state = juce::ValueTree::readFromStream(mis); if (state.isValid()) apvts.replaceState(state);
        }
        currentSampleFolder = juce::File(apvts.state.getProperty("sampleFolder").toString());
        juce::String pat = apvts.state.getProperty("rhythmPattern").toString();
        for (int i = 0; i < 16; ++i) if (i < pat.length()) rhythmPattern[i] = (pat[i] == '1' ? 1.0f : 0.0f);
        auto* sound = dynamic_cast<SonicCryptSound*>(synth.getSound(0).get());
        if (sound) { sound->clear(); for (int i = 0; i < 4; ++i) readAudioLayerFromStream(in, sound, i); }
    }
}

// --- STATE INFORMATION (UPDATED FOR UI SIZE) ---
void SonicCryptChaosAudioProcessor::getStateInformation(juce::MemoryBlock& destData) {
    juce::ScopedLock sl(sampleLoadingLock);
    juce::MemoryOutputStream out(destData, false);
    out.writeString("SCDAW4");

    // Sicherstellen, dass die Variablen im State-Tree aktuell sind
    auto state = apvts.copyState();
    state.setProperty("sampleFolder", currentSampleFolder.getFullPathName(), nullptr);
    juce::String patternString; for (float f : rhythmPattern) patternString += (f > 0.5f ? "1" : "0");
    state.setProperty("rhythmPattern", patternString, nullptr);

    // EXPLIZIT SPEICHERN
    state.setProperty("uiWidth", lastUIWidth, nullptr);
    state.setProperty("uiHeight", lastUIHeight, nullptr);

    state.writeToStream(out);

    auto* sound = dynamic_cast<SonicCryptSound*>(synth.getSound(0).get());
    if (sound) { for (int i = 0; i < 4; ++i) writeAudioLayerToStream(out, sound, i); }
}

void SonicCryptChaosAudioProcessor::setStateInformation(const void* data, int sizeInBytes) {
    juce::ScopedLock sl(sampleLoadingLock);
    juce::MemoryInputStream in(data, sizeInBytes, false);

    if (in.readString() == "SCDAW4") {
        auto newState = juce::ValueTree::readFromStream(in);

        if (newState.isValid()) {
            // FIX: Behalte die aktuelle Größe, falls das geladene Preset keine hat (weil es alt ist)
            int currentW = lastUIWidth;
            int currentH = lastUIHeight;

            // Wenn im neuen State Werte stehen, nimm diese. Sonst behalte currentW/H.
            if (newState.hasProperty("uiWidth")) currentW = newState.getProperty("uiWidth");
            if (newState.hasProperty("uiHeight")) currentH = newState.getProperty("uiHeight");

            // State ersetzen
            apvts.replaceState(newState);

            // Werte wieder in die Variablen und in den State schreiben (damit sie beim nächsten Save da sind)
            lastUIWidth = currentW;
            lastUIHeight = currentH;

            // Zur Sicherheit auch zurück in den Tree schreiben, falls es ein altes Preset war
            apvts.state.setProperty("uiWidth", lastUIWidth, nullptr);
            apvts.state.setProperty("uiHeight", lastUIHeight, nullptr);

            currentSampleFolder = juce::File(newState.getProperty("sampleFolder").toString());
            juce::String pat = newState.getProperty("rhythmPattern").toString();
            for (int i = 0; i < 16; ++i) if (i < pat.length()) rhythmPattern[i] = (pat[i] == '1' ? 1.0f : 0.0f);
        }

        auto* sound = dynamic_cast<SonicCryptSound*>(synth.getSound(0).get());
        if (sound) { sound->clear(); for (int i = 0; i < 4; ++i) readAudioLayerFromStream(in, sound, i); }
    }
}
bool SonicCryptChaosAudioProcessor::hasEditor() const { return true; }
juce::AudioProcessorEditor* SonicCryptChaosAudioProcessor::createEditor() { return new SonicCryptChaosAudioProcessorEditor(*this); }
const juce::String SonicCryptChaosAudioProcessor::getName() const { return "SonicCrypt Chaos"; }
bool SonicCryptChaosAudioProcessor::acceptsMidi() const { return true; }
bool SonicCryptChaosAudioProcessor::producesMidi() const { return false; }
bool SonicCryptChaosAudioProcessor::isMidiEffect() const { return false; }
double SonicCryptChaosAudioProcessor::getTailLengthSeconds() const { return 0.0; }
int SonicCryptChaosAudioProcessor::getNumPrograms() { return 1; }
int SonicCryptChaosAudioProcessor::getCurrentProgram() { return 0; }
void SonicCryptChaosAudioProcessor::setCurrentProgram(int) {}
const juce::String SonicCryptChaosAudioProcessor::getProgramName(int) { return {}; }
void SonicCryptChaosAudioProcessor::changeProgramName(int, const juce::String&) {}
bool SonicCryptChaosAudioProcessor::isBusesLayoutSupported(const BusesLayout&) const { return true; }
void SonicCryptChaosAudioProcessor::valueTreePropertyChanged(juce::ValueTree&, const juce::Identifier&) {}
juce::AudioProcessor* JUCE_CALLTYPE createPluginFilter() { return new SonicCryptChaosAudioProcessor(); }